'use strict';

/* --------------------------------------------------------------
 create_missing_documents.js 2016-10-17
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Create Missing Documents Modal
 *
 * This modal prompts the user for the creation of missing order documents (invoices or packing slips). Make sure
 * that you set the "orderIds", "orderIdsWithoutDocument", "type" data values to the modal before showing it.
 *
 * Example:
 *
 * ```
 * $modal
 *   .data({
 *     orderIds: [...], 
 *     orderIdsWithoutDocument: [...], 
 *     type: 'invoice'
 *   })
 *   .modal('show');
 * ```
 */
gx.controllers.module('create_missing_documents', ['modal'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {
        bindings: {
            createMissingDocumentsCheckbox: $this.find('.create-missing-documents-checkbox')
        }
    };

    /**
     * Proceed Event Constant
     *
     * @type {String}
     */
    var PROCEED_EVENT = 'create_missing_documents:proceed';

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Create missing documents.
     *
     * @param {Number[]} orderIds
     * @param {Number[]} orderIdsWithoutDocument
     * @param {String} type
     */
    function _createMissingDocuments(orderIds, orderIdsWithoutDocument, type) {
        var _$;

        var requests = [];
        var downloadPdfWindow = window.open('about:blank');

        orderIdsWithoutDocument.forEach(function (oID) {
            var data = {
                type: type,
                oID: oID,
                ajax: '1'
            };

            requests.push($.getJSON(jse.core.config.get('appUrl') + '/admin/gm_pdf_order.php', data));
        });

        (_$ = $).when.apply(_$, requests).done(function () {
            return $this.trigger(PROCEED_EVENT, [orderIds, type, downloadPdfWindow]);
        });
    }

    /**
     * Remove order IDs that do not have a document.
     *
     * @param {Number[]} orderIds
     * @param {Number[]} orderIdsWithoutDocument
     * @param {String} type
     */
    function _removeOrderIdsWithoutDocument(orderIds, orderIdsWithoutDocument, type) {
        orderIds.forEach(function (id, index) {
            if (orderIdsWithoutDocument.includes(String(id))) {
                orderIds.splice(index);
            }
        });

        if (!orderIds.length) {
            var title = jse.core.lang.translate('DOWNLOAD_DOCUMENTS', 'admin_orders');
            var message = jse.core.lang.translate('NO_DOCUMENTS_ERROR', 'orders');
            jse.libs.modal.showMessage(title, message);
        } else {
            $this.trigger(PROCEED_EVENT, [orderIds, type]);
        }
    }

    /**
     * On Bulk PDF Modal Confirmation Click
     *
     * Proceed with the generation of the concatenated PDF. This handler will trigger the "create_missing_document"
     */
    function _onOkButtonClick() {
        var createMissingDocuments = module.bindings.createMissingDocumentsCheckbox.get();

        var _$this$data = $this.data(),
            orderIds = _$this$data.orderIds,
            orderIdsWithoutDocument = _$this$data.orderIdsWithoutDocument,
            type = _$this$data.type;

        if (createMissingDocuments) {
            _createMissingDocuments(orderIds, orderIdsWithoutDocument, type);
        } else {
            _removeOrderIdsWithoutDocument(orderIds, orderIdsWithoutDocument, type);
        }

        $this.modal('hide');
    }

    // ------------------------------------------------------------------------
    // INITIALIZE
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $this.on('click', '.btn.ok', _onOkButtonClick);
        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
